<?php
require 'config.php'; // Include database connection

header("Content-Type: application/json");
// Call the function to establish a database connection
$conn = getDbConnection();

$data = json_decode(file_get_contents("php://input"), true);

// Check if request is a POST request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Log incoming POST and FILES data for debugging
    file_put_contents('incident_report_debug.log', "POST Data:\n" . print_r($_POST, true), FILE_APPEND);
    file_put_contents('incident_report_debug.log', "FILES Data:\n" . print_r($_FILES, true), FILE_APPEND);

    // Extract POST data
    $token = $_POST['token'] ?? null;
    $incident_type = $_POST['incident_type'] ?? null;
    $custom_incident = $_POST['custom_incident'] ?? null;
    $incident_description = $_POST['incident_description'] ?? null;
    $corrective_action = $_POST['corrective_action'] ?? null;
    $severity = $_POST['severity'] ?? null;

    $incident_image = $_FILES['incident_image'] ?? null;
    $corrective_image = $_FILES['corrective_image'] ?? null;

    // Validate required fields
    if (empty($token)) {
        http_response_code(400);
        echo json_encode(["message" => "Missing required field: token"]);
        exit;
    }
    if (empty($incident_type)) {
        http_response_code(400);
        echo json_encode(["message" => "Missing required field: incident_type"]);
        exit;
    }
    if (empty($incident_description)) {
        http_response_code(400);
        echo json_encode(["message" => "Missing required field: incident_description"]);
        exit;
    }
    if (empty($corrective_action)) {
        http_response_code(400);
        echo json_encode(["message" => "Missing required field: corrective_action"]);
        exit;
    }
    if (empty($severity)) {
        http_response_code(400);
        echo json_encode(["message" => "Missing required field: severity"]);
        exit;
    }

    // Validate severity
    $validSeverities = ["Low", "Medium", "High"];
    if (!in_array($severity, $validSeverities)) {
        http_response_code(400);
        echo json_encode(["message" => "Invalid severity level"]);
        exit;
    }

    // Validate token
    $stmt = $conn->prepare("SELECT COUNT(*) FROM users WHERE token = ?");
    $stmt->bind_param("s", $token);
    $stmt->execute();
    $stmt->bind_result($count);
    $stmt->fetch();
    $stmt->close();

    if ($count == 0) {
        http_response_code(403);
        echo json_encode(["message" => "Invalid token. Please ensure the token is correct and associated with a registered user."]);
        exit;
    }

    // Handle file uploads
    $incident_image_path = null;
    $corrective_image_path = null;

    $upload_dir = '../uploads/incidents/';
    if (!file_exists($upload_dir) && !mkdir($upload_dir, 0777, true)) {
        http_response_code(500);
        echo json_encode(["message" => "Failed to create upload directory"]);
        exit;
    }

    if ($incident_image) {
        if ($incident_image['error'] !== UPLOAD_ERR_OK) {
            http_response_code(500);
            echo json_encode(["message" => "Error uploading incident image: " . $incident_image['error']]);
            exit;
        }

        $incident_image_path = $upload_dir . uniqid('incident_') . '_' . basename($incident_image['name']);
        if (!move_uploaded_file($incident_image['tmp_name'], $incident_image_path)) {
            http_response_code(500);
            echo json_encode(["message" => "Failed to upload incident image"]);
            exit;
        }
    }

    if ($corrective_image) {
        if ($corrective_image['error'] !== UPLOAD_ERR_OK) {
            http_response_code(500);
            echo json_encode(["message" => "Error uploading corrective image: " . $corrective_image['error']]);
            exit;
        }

        $corrective_image_path = $upload_dir . uniqid('corrective_') . '_' . basename($corrective_image['name']);
        if (!move_uploaded_file($corrective_image['tmp_name'], $corrective_image_path)) {
            http_response_code(500);
            echo json_encode(["message" => "Failed to upload corrective image"]);
            exit;
        }
    }

    // Save the data to the database
    $stmt = $conn->prepare("INSERT INTO incident_reports (token, incident_type, custom_incident, incident_description, incident_image, corrective_action, corrective_image, severity) 
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssssssss", $token, $incident_type, $custom_incident, $incident_description, $incident_image_path, $corrective_action, $corrective_image_path, $severity);

    if ($stmt->execute()) {
        http_response_code(200);
        echo json_encode([
            "message" => "Incident reported successfully",
            "incident_image_path" => $incident_image_path,
            "corrective_image_path" => $corrective_image_path
        ]);
    } else {
        http_response_code(500);
        echo json_encode(["message" => "Database error: " . $stmt->error]);
    }

    $stmt->close();
    $conn->close();
} else {
    http_response_code(405);
    echo json_encode(["message" => "Invalid request method"]);
}

